/*
Copyright © 2005-2018 moonxseed.  All rights reserved.
*/

// Constants
var topShadow = 5;
var bottomShadow = 13;
var leftShadow = 9;
var rightShadow = 9;
var frameTopAdjuster = 10;
var splashFrameWidth = 274;
var splashFrameHeight = 170;
var pinWidth = 40;
var pinHeight = 41;
var orderIndicatorWidth = 64;
var orderIndicatorHeight = 15;
var minImageDisplaySize = 128;
var maxImageDisplaySize = 512;
var minImageDisplaySizeForShortOne = 48;	// Smallest area: 128x48 or 48x128
var maxPinNumber = [11,16,16,16,16,16,16,16,16,16];
var validPinType = [true,true,true,true,true,true,true,true,true,true];
var defaultImageDispSize = 256;
var defaultPinType = 2;
var defaultPinNumber = 10;
var defaultImageIndex = 0;
var defaultImageOrder = 1;

var image = new Image();
var resizedImageWidth = 0;
var resizedImageHeight = 0;
var hasImage = false;
var imageSize = defaultImageDispSize;
var pinType = defaultPinType;
var pinNumber = defaultPinNumber;
var imageIndex = defaultImageIndex;
var imageOrder = defaultImageOrder;	// 0:Random, 1:Ascending, 2:Descending
var indicatorTimer = null;



function initialize()
{
	var canvas = document.getElementById("canvas");
	canvas.width = splashFrameWidth;
	canvas.height = splashFrameHeight;
	canvas.style.width = canvas.width + "px";
	canvas.style.height = canvas.height + "px";

	var usageindicator = document.getElementById("usageindicator");
	usageindicator.innerHTML = "To display an image, put an image file in " + widget.identifier + " folder in the Decor folder. There is the Decor folder in your Pictures folder.<br/>Type 'H' for help."

	if (window.widget) {
		DecorPlugin.prepareFolderForIdentifier(widget.identifier)

		changeImage(0);
		changeImageSize(
				widget.preferenceForKey(makePreferencesKey("Image Size")),
				false, false, false);
		changePin(
				widget.preferenceForKey(makePreferencesKey("Pin Type")),
				widget.preferenceForKey(makePreferencesKey("Pin Number")),
				false, false);
		changeImageOrder(
				widget.preferenceForKey(makePreferencesKey("Image Order")),
				false, false, false);
	}


	setPinImage();

	image.onload = imageLoaded;
	image.onerror = imageError;

	return 0;
}


//---
function makePreferencesKey(key)
{
	return widget.identifier + "-" + key;
}


// Event Handlers

//---
function onhide()
{
	if (hasImage) {
		nextImageIndex(false);
	}
}


//---
function onshow()
{
	if (window.widget) {
		DecorPlugin.prepareFolderForIdentifier(widget.identifier)

		if (hasImage == false) {
			changeImage(0);
		}
	}
}


//---
function onremove()
{
	if (window.widget) {
		DecorPlugin.markFolderAsRemoved()
		widget.setPreferenceForKey(null, makePreferencesKey("URI"));
		widget.setPreferenceForKey(null, makePreferencesKey("Image Index"));
		widget.setPreferenceForKey(null, makePreferencesKey("Image Size"));
		widget.setPreferenceForKey(null, makePreferencesKey("Pin Type"));
		widget.setPreferenceForKey(null, makePreferencesKey("Pin Number"));
		widget.setPreferenceForKey(null, makePreferencesKey("Image Order"));
	}
}

if (window.widget) {
	widget.onremove = onremove;
	widget.onhide = onhide;
	widget.onshow = onshow;
}


//---
function keypress(event)
{
	var handled = true;
	var shift = event.shiftKey;

	switch (event.keyCode) {
		case 8:			// delete key
			image.src = null;
			showSplash();
			break;
		case 72:		// 'H'
		case 104:		// 'h'
			if (widget) {
				widget.system("/usr/bin/open Help/help.html", null);
			}
			break;
		case 48:		// '0'
		case 49:		// '1'
		case 50:		// '2'
		case 51:		// '3'
		case 52:		// '4'
		case 53:		// '5'
		case 54:		// '6'
		case 55:		// '7'
		case 56:		// '8'
		case 57:		// '9'
			if (hasImage) {
				var type = parseInt(String.fromCharCode(event.keyCode));
				changePin(type, "auto", shift, true);
			}
			break;
		case 79:		// 'O'
		case 111:		// 'o'
			if (hasImage && DecorPlugin.count() > 1) {
				changeImageOrder("auto", shift, true, true);
			}
			break;
		case 83:		// 'S'
		case 115:		// 's'
			if (hasImage) {
				changeImageSize("auto", shift, true, true);
			}
			break;
		case 32:		// space bar
			if (hasImage) {
				nextImageIndex(shift);
			}
			break;
		default:
			handled = false;
	}

	if (handled) {
		event.stopPropagation();
		event.preventDefault();
	}
}

if (window.widget) {
	document.addEventListener("keypress", keypress, true);
}


//--- Show the splash image
function showSplash()
{
	hasImage = false;

	var splash = document.getElementById("splash");
	var pin = document.getElementById("pin");
	var canvas = document.getElementById("canvas");
	var usageindicator = document.getElementById("usageindicator");

	splash.style.opacity = 1.0;
	pin.style.opacity = 0.0;
	canvas.style.opacity = 0.0;

	splash.style.width = splashFrameWidth + "px";
	splash.style.height = splashFrameHeight + "px";
	canvas.width = 1;
	canvas.height = 1;
	canvas.style.width = canvas.width + "px";
	canvas.style.height = canvas.height + "px";

	var rorderindicator = document.getElementById("rorderindicator");
	var aorderindicator = document.getElementById("aorderindicator");
	var dorderindicator = document.getElementById("dorderindicator");
	rorderindicator.style.width =
			aorderindicator.style.width =
			dorderindicator.style.width = 1 + "px";
	rorderindicator.style.height =
			aorderindicator.style.height =
			dorderindicator.style.height = 1 + "px";
	rorderindicator.style.left =
			aorderindicator.style.left =
			dorderindicator.style.left = 0 + "px";
	rorderindicator.style.top =
			aorderindicator.style.top =
			dorderindicator.style.top = 0 + "px";

	if (window.widget) {
		window.resizeTo(splashFrameWidth, splashFrameHeight);
		widget.setCloseBoxOffset(9, 17);
		widget.setPreferenceForKey(null, makePreferencesKey("URI"));
	}
}


//---
function adjustPositionAndSize(frameWidth, frameHeight)
{
	// Resize splash
	var splash = document.getElementById("splash");
	splash.style.width = 1 + "px";
	splash.style.height = 1 + "px";

	// Resize canvas
	var canvas = document.getElementById("canvas");
	canvas.width = frameWidth;
	canvas.height = frameHeight;
	canvas.style.width = canvas.width + "px";
	canvas.style.height = canvas.height + "px";

	// Resize window
	if (window.widget) {
		window.resizeTo(frameWidth, frameHeight);
	}

	// Adjust pin position
	var pin = document.getElementById("pin");
	pin.style.left = (frameWidth - pinWidth) / 2  + "px";
	pin.style.top = 1 + "px";

	// Adjust close box position
	if (window.widget) {
		widget.setCloseBoxOffset(leftShadow, topShadow + frameTopAdjuster+2);
	}

	// Resize and adjust order indicator position
	var rorderindicator = document.getElementById("rorderindicator");
	var aorderindicator = document.getElementById("aorderindicator");
	var dorderindicator = document.getElementById("dorderindicator");
	rorderindicator.style.width =
			aorderindicator.style.width =
			dorderindicator.style.width = orderIndicatorWidth + "px";
	rorderindicator.style.height =
			aorderindicator.style.height =
			dorderindicator.style.height = orderIndicatorHeight + "px";
	rorderindicator.style.left =
			aorderindicator.style.left =
			dorderindicator.style.left =
			(resizedImageWidth / 2) + leftShadow - (orderIndicatorWidth / 2) + "px";
	rorderindicator.style.top =
			aorderindicator.style.top =
			dorderindicator.style.top =
			(resizedImageHeight / 2) + topShadow - (orderIndicatorHeight / 2) + "px";
}


//--- Resize
function resize()
{
	resizedImageWidth = image.width;
	resizedImageHeight = image.height;

	// Calculate image size
	var ratio = 0.0;
	if (resizedImageWidth > resizedImageHeight) {
		if (resizedImageWidth > imageSize) {
			ratio = resizedImageHeight / resizedImageWidth;
			resizedImageWidth = imageSize;
			resizedImageHeight = imageSize * ratio;

			if (resizedImageHeight < minImageDisplaySizeForShortOne) {
				resizedImageHeight = minImageDisplaySizeForShortOne;
			}
		}
	}
	else {
		if (resizedImageHeight > imageSize) {
			ratio = resizedImageWidth / resizedImageHeight;
			resizedImageWidth = imageSize * ratio;
			resizedImageHeight = imageSize;

			if (resizedImageWidth < minImageDisplaySizeForShortOne) {
				resizedImageWidth = minImageDisplaySizeForShortOne;
			}
		}
	}

	var frameWidth = resizedImageWidth + leftShadow + rightShadow;
	var frameHeight = resizedImageHeight + topShadow + bottomShadow + frameTopAdjuster;
	adjustPositionAndSize(frameWidth, frameHeight);
}


//--- Draw the picture on the canvas
function drawPicture()
{
	var canvas = document.getElementById("canvas");
	var context = canvas.getContext("2d");

	context.shadowBlur = 10;
	context.shadowColor = "rgba(0, 0, 0, 0.1)";
	context.shadowOffsetX = 0;
	context.shadowOffsetY = 4;
	context.drawImage(image, 0, 0, image.width, image.height,
			leftShadow, topShadow + frameTopAdjuster, resizedImageWidth, resizedImageHeight);
}


//---
function showImage()
{
	var context = canvas.getContext("2d");
	context.clearRect(0, 0, canvas.width, canvas.height);

	resize();

	drawPicture();
}


//---
function imageLoaded()
{
	hasImage = false;

	// Check original image size
	if (image.width < 64 || image.height < 64) {
		showSplash();
		return;
	}

	showImage();

	var splash = document.getElementById("splash");
	var pin = document.getElementById("pin");
	var canvas = document.getElementById("canvas");

	splash.style.opacity = 0.0;
	pin.style.opacity = 1.0;
	canvas.style.opacity = 1.0;

	hasImage = true;
}


//---
function imageError()
{
	showSplash();
	image.src = null;
}


//---
function setPinImage()
{
	var pinNumberStr = pinNumber < 10 ? "0" + pinNumber : pinNumber;
	var pin = document.getElementById("pin");
	pin.src = "Images/Pins/" + pinType + "/" + pinNumberStr + ".png";
}


//---
function changeImage(index)
{
	var imageCount = DecorPlugin.count()

	if (hasImage == false && imageCount == 0) {
		return;
	}

	if (index == undefined || index < 0 || index >= imageCount) {
		index = 0;
	}

	uri = DecorPlugin.imageURIAtIndex(index);

	image.src = uri;
	imageIndex = index;
}


//---
function changePin(type, number, reverse, savePreference)
{
	if (hasImage == false && type == undefined) {
		return;
	}

	if (type == undefined || type < 0 || type > 9) {
		type = defaultPinType;
		number = defaultPinNumber;
	}
	else {
		if (validPinType[type] == false) {
			return;
		}

		if (number == "auto") {
			if (type != pinType) {
				number = 1;
			}
			else {
				if (reverse) {
					number = pinNumber - 1;
					if (number < 1) {
						number = maxPinNumber[type];
					}
				}
				else {
					number = pinNumber + 1;
					if (number > maxPinNumber[type]) {
						number = 1;
					}
				}
			}
		}
		else if (number == undefined ||
				number < 1 || number > maxPinNumber[type]) {
			number = defaultPinNumber;
		}
	}

	pinType = type;
	pinNumber = number;

	if (savePreference) {
		if (window.widget) {
			widget.setPreferenceForKey(pinType, makePreferencesKey("Pin Type"));
			widget.setPreferenceForKey(pinNumber, makePreferencesKey("Pin Number"));
		}
	}

	setPinImage();
}


//---
function changeImageSize(size, reverse, resize, savePreference)
{
	if (hasImage == false && size == undefined) {
		return;
	}

	if (size == "auto") {
		if (reverse) {
			size = ((imageSize / 32) - 1) * 32;
			if (size < minImageDisplaySize) {
				size = maxImageDisplaySize;
			}
		}
		else {
			size = ((imageSize / 32) + 1) * 32;
			if (size > maxImageDisplaySize) {
				size = minImageDisplaySize;
			}
		}
	}
	else if (size == undefined ||
			size < minImageDisplaySize || size > maxImageDisplaySize) {
		size = defaultImageDispSize;
	}

	imageSize = size;

	if (savePreference) {
		if (window.widget) {
			widget.setPreferenceForKey(imageSize, makePreferencesKey("Image Size"));
		}
	}

	// Resize
	if (resize) {
		showImage();
	}
}


//---
function nextImageIndex(reverse)
{
	var index;
	if (imageIndex != undefined) {
		if (imageOrder == 0) {
			index = DecorPlugin.count() - 1;
			index = Math.round(index * Math.random());
		}
		else {
			if (imageOrder == 2) {
				reverse = !reverse;
			}

			if (reverse) {
				index = imageIndex - 1;
				if (index < 0) {
					index = DecorPlugin.count() - 1;
				}
			}
			else {
				index = imageIndex + 1;
				if (index >= DecorPlugin.count()) {
					index = 0;
				}
			}
		}

		changeImage(index);
	}
}


//---
function changeImageOrder(order, shift, feedback, savePreference)
{
	if (hasImage == false && order == undefined) {
		return;
	}

	if (order == "auto") {
		var val = shift ? -1 : 1;
		order = (imageOrder+val) % 3;
		if (order < 0) {
			order = 2;
		}
	}
	else if (order == undefined || order < 0 || order > 2) {
		order = 1;
	}

	imageOrder = order;

	if (savePreference) {
		if (window.widget) {
			widget.setPreferenceForKey(imageOrder, makePreferencesKey("Image Order"));
		}
	}

	if (feedback) {
		feedbackImageOrder();
	}
}


//---
function feedbackImageOrder()
{
	var rorderindicator = document.getElementById("rorderindicator");
	var aorderindicator = document.getElementById("aorderindicator");
	var dorderindicator = document.getElementById("dorderindicator");
	rorderindicator.style.opacity = 0.0;
	aorderindicator.style.opacity = 0.0;
	dorderindicator.style.opacity = 0.0;
	switch (imageOrder) {
	case 0:
		rorderindicator.style.opacity = 1.0;
		break;
	case 1:
		aorderindicator.style.opacity = 1.0;
		break;
	case 2:
		dorderindicator.style.opacity = 1.0;
		break;
	}

	if (indicatorTimer != null) {
		clearInterval(indicatorTimer);
		indicatorTimer = null;
	}

	if (fader.timer != null) {
		clearInterval(fader.timer);
		fader.timer = null;
	}

	indicatorKicker();
}


//---
function indicatorKicker()
{
	indicatorTimer = setInterval ("indicatorFadeOut();", 777);
}


//---
function indicatorFadeOut()
{
	if (indicatorTimer != null) {
		clearInterval(indicatorTimer);
		indicatorTimer = null;
	}

	var starttime = (new Date).getTime() - 13; // set it back one frame

	fader.duration = 500;
	fader.starttime = starttime;
	switch (imageOrder) {
	case 0:
		fader.firstElement = document.getElementById("rorderindicator");
		break;
	case 1:
		fader.firstElement = document.getElementById("aorderindicator");
		break;
	case 2:
		fader.firstElement = document.getElementById("dorderindicator");
		break;
	}
	fader.timer = setInterval("fade();", 13);
	fader.from = fader.now = 1.0;
	fader.to = 0.0;
	fade();
}



var fader = {
	duration:0,
	starttime:0,
	to:1.0,
	now:0.0,
	from:0.0,
	firstElement:null,
	timer:null
};

function fade()
{
	var time = (new Date).getTime();

	var T = limit_3(time-fader.starttime, 0, fader.duration);

	if (T >= fader.duration || !hasImage) {
		clearInterval(fader.timer);
		fader.timer = null;
		fader.now = fader.to;
	}
	else {
		var ease = 0.5 - (0.5 * Math.cos(Math.PI * T / fader.duration));
		fader.now = computeNextFloat(fader.from, fader.to, ease);
	}

	fader.firstElement.style.opacity = fader.now;
}
